<?php
/**
 * Handle all svg icons in one place.
 *
 * @package WPCode
 */

/**
 * Get an SVG icon by name with width, height and viewbox options.
 *
 * @param string $name The name of the icon.
 * @param int    $width The width.
 * @param int    $height The height.
 * @param string $viewbox The viewbox, will be auto-built from width and height if not set.
 *
 * @return string
 */
function wsrw_get_icon( $name, $width = 20, $height = 20, $viewbox = '' ) {
	$icons = wsrw_icons();

	if ( ! isset( $icons[ $name ] ) ) {
		return '';
	}

	if ( empty( $viewbox ) ) {
		$viewbox = sprintf( '0 0 %1$s %2$s', $width, $height );
	}

	return sprintf(
		'<svg class="wsrw-icon wsrw-icon-%5$s" width="%1$s" height="%2$s" viewBox="%3$s" fill="none" xmlns="http://www.w3.org/2000/svg">%4$s</svg>',
		$width,
		$height,
		$viewbox,
		$icons[ $name ],
		$name
	);
}

/**
 * Echo an icon in a safe mode.
 *
 * @param string $name The name of the icon.
 * @param int    $width The width.
 * @param int    $height The height.
 * @param string $viewbox The viewbox, will be auto-built from width and height if not set.
 *
 * @return void
 */
function wsrw_icon( $name, $width = 20, $height = 20, $viewbox = '' ) {
	$icon = wsrw_get_icon( $name, $width, $height, $viewbox );

	if ( ! empty( $icon ) ) {
		echo wp_kses(
			$icon,
			wsrw_get_icon_allowed_tags()
		);
	}
}

/**
 * Get the allowed tags for svg icons, to be used in wp_kses.
 *
 * @return array
 * @see wsrw_icon()
 * @see wp_kses()
 */
function wsrw_get_icon_allowed_tags() {
	return array(
		'svg'      => array(
			'class'           => true,
			'aria-hidden'     => true,
			'aria-labelledby' => true,
			'role'            => true,
			'xmlns'           => true,
			'width'           => true,
			'height'          => true,
			'viewbox'         => true,
		),
		'g'        => array(
			'fill'      => true,
			'clip-path' => true,
			'transform' => true,
			'stroke'    => true,
		),
		'title'    => array( 'title' => true ),
		'path'     => array(
			'd'         => true,
			'fill'      => true,
			'fill-rule' => true,
			'clip-rule' => true,
		),
		'circle'   => array(
			'cx'           => true,
			'cy'           => true,
			'r'            => true,
			'stroke'       => true,
			'stroke-width' => true,
			'fill'         => true,
		),
		'rect'     => array(
			'x'      => true,
			'y'      => true,
			'width'  => true,
			'height' => true,
			'fill'   => true,
		),
		'clipPath' => array(
			'id' => true,
		),
		'defs'     => array(),
	);
}

/**
 * Get the whole array of WPCode SVG icons.
 *
 * @return array
 */
function wsrw_icons() {
	return (array) apply_filters(
		'wsrw_icons',
		array(
			'logo'          => '<path fill="#0265FC" d="M17.1 2.2c-6.9 2.6-12.3 8.1-15 15.2C0 23 0 23.6 0 128c0 104.4 0 105 2.1 110.6 2.7 7.2 8.1 12.6 15.3 15.3C23 256 23.6 256 128 256c104.4 0 105 0 110.6-2.1 7.2-2.7 12.6-8.1 15.3-15.3 2.1-5.6 2.1-6.2 2.1-110.6 0-104.4 0-105-2.1-110.6-2.7-7.2-8.1-12.6-15.3-15.3C233 0 232.4 0 127.7.1 24 .1 22.4.1 17.1 2.2zm134.2 65.2c3.4 1.4 48 45.5 50.3 49.7 1.8 3.2 1.8 13.1 0 17.2-2 4.5-47.4 49.4-50.9 50.2-10.3 2.6-19.2-8.1-14.3-17.3.8-1.5 10.3-11.5 21-22.2l19.6-19.5-19.4-19.5c-10.6-10.7-20.1-20.6-21-22-3.9-5.9.1-14.6 7.9-17.2 3-.9 3.3-.9 6.8.6zM114.9 92c11.7 2.2 20.3 8.5 25.3 18.7 8.6 17.6 2 37.5-15.7 46.6-5.2 2.6-7.1 3.1-14 3.1-4.4.1-9.1-.2-10.5-.6-2.7-.8-2.4-1-21.1 16.9-6.7 6.3-8.1 7.3-11.3 7.3-9.5 0-15.7-8.2-12.3-16.1.8-1.9 6.5-8.3 12.5-14.3l11.1-10.9-1.7-3.1c-1.3-2.4-1.7-5.6-1.7-13.6 0-9.1.3-11.1 2.4-15 5-9.6 15.2-17.3 25-19 6.6-1.1 6.3-1.1 12 0z"/><path fill="#0265FC" d="M103 112.5c-10.5 4.5-11.6 19.8-2 26.3 3 2 4.4 2.3 9.4 2 5.6-.4 6.2-.8 9.8-4.8 3.3-3.7 3.8-4.9 3.8-9.2 0-2.7-.7-6.2-1.5-7.8-3.3-6.4-12.6-9.5-19.5-6.5z"/>',
			'logo-text'     => '<path fill="#11293e" stroke-width="0" d="M47.03 11.53c0-3.89.91-6.55 2.72-8.5 1.8-1.97 4.37-2.94 7.67-2.94 1.54 0 3.03.32 4.32.95a8.21 8.21 0 0 1 3.14 2.55c.17.2.21.43.21.69 0 .32-.11.54-.32.74-.2.2-.43.32-.74.32-.37 0-.63-.17-.86-.43-.6-.8-1.38-1.49-2.46-2.03-1-.54-2.14-.8-3.37-.8-2.6 0-4.58.8-5.97 2.4-1.38 1.6-2.08 3.78-2.08 6.98s.69 5.58 2.03 7.07c1.34 1.49 3.35 2.29 6.03 2.29 1.17 0 2.23-.2 3.31-.69 1.01-.48 1.86-1.12 2.51-2.03.2-.26.48-.43.86-.43.32 0 .54.11.74.32.2.2.32.48.32.74 0 .2-.06.43-.21.63-.74 1.06-1.77 1.92-3.14 2.55-1.38.63-2.83.95-4.37.95-3.35 0-5.92-.95-7.68-2.83-1.75-1.8-2.64-4.52-2.64-8.5ZM69.24 20.95c-1.23-1.32-1.86-3.35-1.86-6.01s.6-4.63 1.81-5.95c1.23-1.32 2.94-1.97 5.17-1.97s3.94.63 5.17 1.97c1.23 1.32 1.8 3.29 1.8 5.95s-.6 4.69-1.8 6.01c-1.23 1.32-2.94 2.03-5.17 2.03s-3.89-.69-5.12-2.03Zm.34-6.01c0 4.04 1.6 6.07 4.8 6.07s4.8-2.03 4.8-6.07-1.6-6.07-4.8-6.07-4.8 2.08-4.8 6.07ZM83.39 14.99c0-5.32 2.23-7.67 6.66-7.67 1.01 0 1.92.2 2.72.63.8.43 1.43.91 1.8 1.54V1.15c0-.32.11-.54.32-.74.2-.2.43-.32.74-.32s.54.11.74.32c.21.2.32.43.32.74v15.27c0 2.12-.59 3.72-1.8 4.84-1.17 1.12-2.77 1.69-4.74 1.69-2.18 0-3.89-.69-5.06-2.03-1.12-1.3-1.69-3.27-1.69-5.93Zm2.12 0c0 4.04 1.54 6.07 4.69 6.07 1.34 0 2.4-.37 3.2-1.17.8-.8 1.17-1.92 1.17-3.46v-4.26c-.06-.06-.11-.17-.17-.32-.06-.17-.21-.37-.48-.69-.26-.32-.54-.63-.86-.91-.32-.26-.74-.48-1.23-.69-.54-.2-1.06-.32-1.65-.32-1.55 0-2.66.48-3.52 1.49-.78 1-1.15 2.23-1.15 4.26ZM99.26 14.94c0-2.55.6-4.52 1.81-5.9 1.21-1.38 2.88-2.08 5.06-2.08.86 0 1.66.11 2.29.32.69.2 1.28.48 1.71.86.43.37.86.74 1.17 1.23.32.43.6.91.74 1.43.17.54.26.95.37 1.38.11.43.11.86.11 1.28v.74c0 .17-.06.37-.17.63-.06.26-.21.43-.43.54a1.5 1.5 0 0 1-.74.17h-9.68c0 1.66.43 2.98 1.34 3.94.91.95 2.14 1.49 3.68 1.49.86 0 1.6-.11 2.18-.37.6-.26 1.23-.58 1.81-1.06.21-.17.43-.2.6-.2.26 0 .48.11.69.26.17.2.26.43.26.69s-.11.54-.37.74c-.21.2-.54.48-.91.74s-.97.48-1.77.74-1.66.37-2.57.37c-2.29 0-4.05-.63-5.28-1.97-1.25-1.25-1.9-3.26-1.9-5.97Zm2.19-1.23h8.95c0-.63-.06-1.23-.2-1.75-.11-.54-.32-1-.63-1.49s-.74-.86-1.34-1.12c-.6-.26-1.28-.43-2.08-.43-1.43 0-2.51.43-3.37 1.28-.78.84-1.26 2.01-1.32 3.5ZM0 2.01c0-.54.2-1 .6-1.38.37-.37.85-.57 1.39-.57.48 0 .91.17 1.23.43.32.26.54.58.63.91l3.89 13.93 3.78-13.88c.11-.43.32-.74.69-1 .37-.26.74-.37 1.12-.37.43 0 .8.11 1.12.37.32.26.6.58.69 1l3.78 13.88 3.89-13.99c.11-.32.32-.63.63-.91.32-.26.69-.43 1.23-.43s.97.17 1.34.54c.37.37.54.8.54 1.32 0 .2-.06.43-.11.58l-5.71 19c-.11.37-.32.74-.69.95-.32.26-.69.37-1.12.37s-.8-.11-1.12-.37c-.32-.26-.54-.58-.63-1L13.34 7.77 9.51 21.54c-.11.43-.37.74-.69 1-.32.26-.74.37-1.17.37s-.8-.11-1.12-.37c-.32-.26-.6-.58-.69-.95L.11 2.6C.05 2.43 0 2.23 0 2ZM29.26 21.06V2.01c0-.54.2-.95.54-1.32.37-.37.86-.58 1.4-.58h6.34c2.08 0 3.74.58 5 1.75 1.28 1.17 1.97 2.87 1.97 5.06s-.63 3.89-1.92 5.06-2.98 1.75-5 1.75h-4.43v7.35c0 .54-.17 1-.54 1.38-.37.37-.8.54-1.4.54s-1-.2-1.38-.58c-.43-.41-.6-.84-.6-1.36Zm3.86-10.81h4.11c1.06 0 1.86-.32 2.46-.91.6-.58.86-1.43.86-2.49s-.26-1.92-.86-2.49c-.6-.58-1.4-.86-2.4-.86h-4.11v6.75h-.06Z"/>',
			'close'         => '<path d="M14.5649 1.41L13.1587 0L7.58348 5.59L2.00831 0L0.602051 1.41L6.17722 7L0.602051 12.59L2.00831 14L7.58348 8.41L13.1587 14L14.5649 12.59L8.98974 7L14.5649 1.41Z" fill="#8C8F9A"/>',
			'upload'        => '<path d="M10.5 8.25V10.5H1.5V8.25H0V10.5C0 11.325 0.675 12 1.5 12H10.5C11.325 12 12 11.325 12 10.5V8.25H10.5ZM2.25 3.75L3.3075 4.8075L5.25 2.8725V9H6.75V2.8725L8.6925 4.8075L9.75 3.75L6 0L2.25 3.75Z" fill="#777777"/>',
			'help'          => '<path fill-rule="evenodd" clip-rule="evenodd" d="M2.16667 9.99984C2.16667 5.39984 5.90001 1.6665 10.5 1.6665C15.1 1.6665 18.8333 5.39984 18.8333 9.99984C18.8333 14.5998 15.1 18.3332 10.5 18.3332C5.90001 18.3332 2.16667 14.5998 2.16667 9.99984ZM11.3333 13.3332V14.9998H9.66667V13.3332H11.3333ZM10.5 16.6665C6.82501 16.6665 3.83334 13.6748 3.83334 9.99984C3.83334 6.32484 6.82501 3.33317 10.5 3.33317C14.175 3.33317 17.1667 6.32484 17.1667 9.99984C17.1667 13.6748 14.175 16.6665 10.5 16.6665ZM7.16667 8.33317C7.16667 6.4915 8.65834 4.99984 10.5 4.99984C12.3417 4.99984 13.8333 6.4915 13.8333 8.33317C13.8333 9.40227 13.175 9.9776 12.534 10.5378C11.9259 11.0692 11.3333 11.587 11.3333 12.4998H9.66667C9.66667 10.9821 10.4518 10.3803 11.142 9.85123C11.6835 9.43618 12.1667 9.06585 12.1667 8.33317C12.1667 7.4165 11.4167 6.6665 10.5 6.6665C9.58334 6.6665 8.83334 7.4165 8.83334 8.33317H7.16667Z" fill="#777777"/>',
			'facebook'      => '<path fill="#A7AAAD" d="M16 8.05A8.02 8.02 0 0 0 8 0C3.58 0 0 3.6 0 8.05A8 8 0 0 0 6.74 16v-5.61H4.71V8.05h2.03V6.3c0-2.02 1.2-3.15 3-3.15.9 0 1.8.16 1.8.16v1.98h-1c-1 0-1.31.62-1.31 1.27v1.49h2.22l-.35 2.34H9.23V16A8.02 8.02 0 0 0 16 8.05Z"/>',
			'twitter'       => '<path fill="#A7AAAD" d="M15.27 4.43A7.4 7.4 0 0 0 17 2.63c-.6.27-1.3.47-2 .53a3.41 3.41 0 0 0 1.53-1.93c-.66.4-1.43.7-2.2.87a3.5 3.5 0 0 0-5.96 3.2 10.14 10.14 0 0 1-7.2-3.67C.86 2.13.7 2.73.7 3.4c0 1.2.6 2.26 1.56 2.89a3.68 3.68 0 0 1-1.6-.43v.03c0 1.7 1.2 3.1 2.8 3.43-.27.06-.6.13-.9.13a3.7 3.7 0 0 1-.66-.07 3.48 3.48 0 0 0 3.26 2.43A7.05 7.05 0 0 1 0 13.24a9.73 9.73 0 0 0 5.36 1.57c6.42 0 9.91-5.3 9.91-9.92v-.46Z"/>',
			'lock'          => '<path d="M19 9.33333H17.6666V6.66667C17.6666 2.98667 14.68 0 11 0C7.31998 0 4.33331 2.98667 4.33331 6.66667V9.33333H2.99998C1.53331 9.33333 0.333313 10.5333 0.333313 12V25.3333C0.333313 26.8 1.53331 28 2.99998 28H19C20.4666 28 21.6666 26.8 21.6666 25.3333V12C21.6666 10.5333 20.4666 9.33333 19 9.33333ZM6.99998 6.66667C6.99998 4.45333 8.78665 2.66667 11 2.66667C13.2133 2.66667 15 4.45333 15 6.66667V9.33333H6.99998V6.66667ZM19 25.3333H2.99998V12H19V25.3333ZM11 21.3333C12.4666 21.3333 13.6666 20.1333 13.6666 18.6667C13.6666 17.2 12.4666 16 11 16C9.53331 16 8.33331 17.2 8.33331 18.6667C8.33331 20.1333 9.53331 21.3333 11 21.3333Z" fill="#8A8A8A"></path>',
			'arrow-right'   => '<path fill-rule="evenodd" clip-rule="evenodd" d="M10.5119 4.43057C10.1974 4.70014 10.161 5.17361 10.4306 5.48811L16.0122 12L10.4306 18.5119C10.161 18.8264 10.1974 19.2999 10.5119 19.5695C10.8264 19.839 11.2999 19.8026 11.5694 19.4881L17.5694 12.4881C17.8102 12.2072 17.8102 11.7928 17.5694 11.5119L11.5694 4.51192C11.2999 4.19743 10.8264 4.161 10.5119 4.43057Z" fill="#fff"/>',
			'arrow-left'    => '<path fill-rule="evenodd" clip-rule="evenodd" d="M13.4881 4.43057C13.8026 4.70014 13.839 5.17361 13.5694 5.48811L7.98781 12L13.5694 18.5119C13.839 18.8264 13.8026 19.2999 13.4881 19.5695C13.1736 19.839 12.7001 19.8026 12.4306 19.4881L6.43056 12.4881C6.18981 12.2072 6.18981 11.7928 6.43056 11.5119L12.4306 4.51192C12.7001 4.19743 13.1736 4.161 13.4881 4.43057Z" fill="#fff"/>',
			'd-arrow-right' => '<path fill-rule="evenodd" clip-rule="evenodd" d="M10.5119 4.43057C10.1974 4.70014 10.161 5.17361 10.4306 5.48811L16.0122 12L10.4306 18.5119C10.161 18.8264 10.1974 19.2999 10.5119 19.5695C10.8264 19.839 11.2999 19.8026 11.5694 19.4881L17.5694 12.4881C17.8102 12.2072 17.8102 11.7928 17.5694 11.5119L11.5694 4.51192C11.2999 4.19743 10.8264 4.161 10.5119 4.43057Z" fill="#fff"/><path d="M6.25 5.00005C6.25 4.68619 6.44543 4.40553 6.73979 4.29664C7.03415 4.18774 7.36519 4.27366 7.56944 4.51196L13.5694 11.512C13.8102 11.7928 13.8102 12.2073 13.5694 12.4881L7.56944 19.4881C7.36519 19.7264 7.03415 19.8124 6.73979 19.7035C6.44543 19.5946 6.25 19.3139 6.25 19L6.25 5.00005Z" fill="#fff"/>',
			'd-arrow-left'  => '<path fill-rule="evenodd" clip-rule="evenodd" d="M13.4881 4.43057C13.8026 4.70014 13.839 5.17361 13.5694 5.48811L7.98781 12L13.5694 18.5119C13.839 18.8264 13.8026 19.2999 13.4881 19.5695C13.1736 19.839 12.7001 19.8026 12.4306 19.4881L6.43056 12.4881C6.18981 12.2072 6.18981 11.7928 6.43056 11.5119L12.4306 4.51192C12.7001 4.19743 13.1736 4.161 13.4881 4.43057Z" fill="#fff"/><path d="M17.75 5.00005C17.75 4.68619 17.5546 4.40553 17.2602 4.29664C16.9658 4.18774 16.6348 4.27366 16.4306 4.51196L10.4306 11.512C10.1898 11.7928 10.1898 12.2073 10.4306 12.4881L16.4306 19.4881C16.6348 19.7264 16.9658 19.8124 17.2602 19.7035C17.5546 19.5946 17.75 19.3139 17.75 19L17.75 5.00005Z" fill="#fff"/>',
		)
	);
}

/**
 * Returns the site domain.
 *
 * @return string
 */
function wsrw_get_site_domain() {
	return wp_parse_url( home_url(), PHP_URL_HOST );
}
